<?php

namespace App\Http\Controllers\Api\V1\Auth;

use App\CentralLogics\Helpers;
use App\Http\Controllers\Controller;
use App\Models\DeliveryMan;
use Carbon\CarbonInterval;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;

class DeliveryManPasswordResetController extends Controller
{
    public function __construct(private DeliveryMan $deliveryMan)
    {
    }

    public function forgotDeliveryManPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => Helpers::error_processor($validator)], 403);
        }

        $deliveryMan = $this->deliveryMan->where(['email' => $request['email']])->first();

        if ($deliveryMan) {
            $otp_interval_time= Helpers::get_business_settings('otp_resend_time') ?? 60;
            $password_verification_data= DB::table('password_resets')->where('email_or_phone', $request['email'])->first();

            if(isset($password_verification_data) &&  Carbon::parse($password_verification_data->created_at)->DiffInSeconds() < $otp_interval_time){
                $time= $otp_interval_time - Carbon::parse($password_verification_data->created_at)->DiffInSeconds();

                $errors = [];
                $errors [] = [
                    'code' => 'otp',
                    'message' => translate('please_try_again_after_') . $time . ' ' . translate('seconds')
                ];

                return response()->json(['errors' => $errors], 403);
            }

            $token = (env('APP_MODE') == 'live') ? rand(100000, 999999) : 123456;

            DB::table('password_resets')->updateOrInsert(['email_or_phone' => $request['email']], [
                'email_or_phone' => $request['email'],
                'token' => $token,
                'created_at' => now(),
                'otp_hit_count' => 0,
                'is_temp_blocked' => 0,
                'temp_block_time' => null,
            ]);

            try {
                $emailServices = Helpers::get_business_settings('mail_config');
                if (isset($emailServices['status']) && $emailServices['status'] == 1) {
                    Mail::to($deliveryMan['email'])->send(new \App\Mail\PasswordResetMail($token));
                }
            } catch (\Exception $exception) {
                return response()->json(['errors' => [
                    ['code' => 'config-missing', 'message' => 'Email configuration issue.']
                ]], 400);
            }

            return response()->json(['message' => 'Email sent successfully.'], 200);
        }

        return response()->json(['errors' => [
            ['code' => 'not-found', 'message' => 'DeliveryMan not found!']
        ]], 401);
    }

    public function verifyDeliveryManToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'reset_token' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => Helpers::error_processor($validator)], 403);
        }

        $maxOtpHit       = Helpers::get_business_settings('maximum_otp_hit') ?? 5;
        $otpWindow       = Helpers::get_business_settings('otp_resend_time') ?? 60;
        $tempBlockTime   = Helpers::get_business_settings('temporary_block_time') ?? 600;

        $verify = DB::table('password_resets')
            ->where('email_or_phone', $request->email)
            ->where('token', $request->reset_token)
            ->first();

        if ($verify) {
            if ($verify->temp_block_time && Carbon::parse($verify->temp_block_time)->diffInSeconds() <= $tempBlockTime) {
                $remaining = $tempBlockTime - Carbon::parse($verify->temp_block_time)->diffInSeconds();

                return response()->json([
                    'errors' => [[
                        'code' => 'otp',
                        'message' => translate('please_try_again_after_') .
                            CarbonInterval::seconds($remaining)->cascade()->forHumans()
                    ]]
                ], 403);
            }

            return response()->json(['message' => 'Token verified successfully'], 200);
        }

        $record = DB::table('password_resets')
            ->where('email_or_phone', $request->email)
            ->first();


        if ($record) {
            if ($record->temp_block_time && Carbon::parse($record->temp_block_time)->diffInSeconds() <= $tempBlockTime) {
                $remaining = $tempBlockTime - Carbon::parse($record->temp_block_time)->diffInSeconds();

                return response()->json([
                    'errors' => [[
                        'code' => 'otp_block_time',
                        'message' => translate('please_try_again_after_') .
                            CarbonInterval::seconds($remaining)->cascade()->forHumans()
                    ]]
                ], 403);
            }

            if ($record->is_temp_blocked && Carbon::parse($record->created_at)->diffInSeconds() >= $otpWindow) {
                DB::table('password_resets')
                    ->where('email_or_phone', $request->email)
                    ->update([
                        'otp_hit_count' => 0,
                        'is_temp_blocked' => 0,
                        'temp_block_time' => null,
                        'created_at' => now(),
                    ]);
            }

            if ($record->otp_hit_count >= $maxOtpHit && Carbon::parse($record->created_at)->diffInSeconds() < $otpWindow && !$record->is_temp_blocked) {
                DB::table('password_resets')
                    ->where('email_or_phone', $request->email)
                    ->update([
                        'is_temp_blocked' => 1,
                        'temp_block_time' => now(),
                        'created_at' => now(),
                    ]);

                return response()->json([
                    'errors' => [[
                        'code' => 'otp_block_time',
                        'message' => translate('please_try_again_after_') .
                            CarbonInterval::seconds($tempBlockTime)->cascade()->forHumans()
                    ]]
                ], 403);
            }
        }

        DB::table('password_resets')->updateOrInsert(
            ['email_or_phone' => $request->email],
            [
                'otp_hit_count' => DB::raw('COALESCE(otp_hit_count,0)+1'),
                'created_at' => now(),
                'temp_block_time' => null,
            ]
        );

        return response()->json([
            'errors' => [[
                'code' => 'invalid',
                'message' => 'OTP is not matched.'
            ]]
        ], 401);
    }

    public function resetDeliveryManPassword(Request $request) {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'reset_token' => 'required',
            'password' => 'required|min:8',
            'confirm_password' => 'required|same:password',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => Helpers::error_processor($validator)], 403);
        }

        $data = DB::table('password_resets')
            ->where('email_or_phone', $request->email)
            ->where('token', $request->reset_token)
            ->first();

        if (!$data) {
            return response()->json(['errors' => [
                ['code' => 'invalid', 'message' => 'Invalid token.']
            ]], 401);
        }

        $deliveryMan = $this->deliveryMan->where('email', $request->email)->first();

        if (!$deliveryMan) {
            return response()->json(['errors' => [
                ['code' => 'not_found', 'message' => 'Delivery man not found']
            ]], 401);
        }

        $deliveryMan->password = bcrypt($request->password);
        $deliveryMan->save();

        DB::table('password_resets')
            ->where('email_or_phone', $request->email)
            ->where('token', $request->reset_token)
            ->delete();

        return response()->json(['message' => 'Password changed successfully.'], 200);
    }
}
