<?php

namespace App\Http\Controllers\Admin;

use App\CentralLogics\Helpers;
use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\DeliveryMan;
use App\Models\DMReview;
use App\Models\Order;
use App\Traits\UploadSizeHelperTrait;
use Brian2694\Toastr\Facades\Toastr;
use Carbon\Carbon;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Propaganistas\LaravelPhone\Rules\Phone;
use Rap2hpoutre\FastExcel\FastExcel;

class DeliveryManController extends Controller
{
    use UploadSizeHelperTrait;

    public function __construct(
        private DeliveryMan $delivery_man,
        private DMReview    $dm_review,
        private Branch      $branch,
        private Order       $order
    )
    {
        $this->initUploadLimits();
    }

    /**
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {
        return view('admin-views.delivery-man.index');
    }

    /**
     * @param Request $request
     * @return Application|Factory|View
     */
    public function list(Request $request): Factory|View|Application
    {
        $data['deliveryMenList'] = $this->getDeliveryMenList($request)
            ->paginate((int)($request->per_page ?? Helpers::pagination_limit()))
            ->appends($request->all());

        $stats = $this->delivery_man
            ->where('application_status', 'approved')
            ->selectRaw('COUNT(*) as total, SUM(is_active = 1) as active, SUM(is_active = 0) as inactive')
            ->first();

        $data['totalDeliveryMen'] = $stats->total;
        $data['activeDeliveryMen'] = $stats->active;
        $data['inactiveDeliveryMen'] = $stats->inactive;

        return view('admin-views.delivery-man.list', $data);
    }

    /**
     * @param Request $request
     * @return Application|Factory|View
     */
    public function reviewsList(Request $request): Factory|View|Application
    {
        $perPage = (int)$request->query('per_page', Helpers::getPagination());
        $queryParams = ['per_page' => $perPage];

        $search = $request->query('search');

        // Start query
        $query = $this->dm_review;

        // Apply search filter
        if ($search) {
            $queryParams['search'] = $search;
            $query = $query->whereHas('delivery_man', function ($q) use ($search) {
                $q->where(function ($q2) use ($search) {
                    $q2->where('f_name', 'like', "%{$search}%")
                        ->orWhere('l_name', 'like', "%{$search}%");
                });
            });
        }

        $reviews = $query->with(['delivery_man', 'customer'])->latest()->paginate($perPage)->appends($queryParams);
        return view('admin-views.delivery-man.reviews-list', compact('reviews', 'search', 'perPage'));
    }

    /**
     * @param $id
     * @return Application|Factory|View
     */
    public function preview($id): Factory|View|Application
    {
        $deliveryMan = $this->delivery_man->with(['reviews'])->where(['id' => $id])->first();
        $reviews = $this->dm_review->where(['delivery_man_id' => $id])->latest()->paginate(20);
        return view('admin-views.delivery-man.view', compact('deliveryMan', 'reviews'));
    }

    /**
     * @param Request $request
     * @return Application|RedirectResponse|Redirector
     */
    public function store(Request $request): Redirector|Application|RedirectResponse
    {
        $check = $this->validateUploadedFile($request, ['image', 'identity_image']);
        if ($check !== true) {
            return $check;
        }

        $request->validate([
            'f_name' => 'required',
            'email' => 'required|regex:/(.+)@(.+)\.(.+)/i|unique:delivery_men',
            'phone' => ['required', 'unique:delivery_men', new Phone()],
            'password' => 'required|min:8',
            'password_confirmation' => 'required_with:password|same:password|min:8',
            'image' => 'sometimes|image|max:' . $this->maxImageSizeKB . '|mimes:' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
            'identity_image' => 'sometimes|array',
            'identity_image.*' => 'image|max:' . $this->maxImageSizeKB . '|mimes:' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
        ], [
            'f_name.required' => translate('First name is required!'),
            'email.required' => translate('Email is required!'),
            'email.unique' => translate('Email must be unique!'),
            'phone.required' => translate('Phone is required!'),
            'phone.unique' => translate('Phone must be unique!'),
            'image.mimes' => 'Image must be a file of type: ' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
            'image.max' => translate('Image size must be below ' . $this->maxImageSizeReadable),
            'identity_image.*.mimes' => 'Identity image must be a file of type: ' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
            'identity_image.*.max' => translate('Identity image size must be below ' . $this->maxImageSizeReadable),
        ]);

        if (!empty($request->file('image'))) {
            $image_name = Helpers::upload('delivery-man/', APPLICATION_IMAGE_FORMAT, $request->file('image'));
        } else {
            $image_name = 'def.png';
        }

        $id_img_names = [];
        if (!empty($request->file('identity_image'))) {
            foreach ($request->identity_image as $img) {
                $identity_image = Helpers::upload('delivery-man/', APPLICATION_IMAGE_FORMAT, $img);
                $id_img_names[] = $identity_image;
            }
            $identity_image = json_encode($id_img_names);
        } else {
            $identity_image = json_encode([]);
        }

        $dm = $this->delivery_man;
        $dm->f_name = $request->f_name;
        $dm->l_name = $request->l_name;
        $dm->email = $request->email;
        $dm->phone = $request->phone;
        $dm->identity_number = $request->identity_number;
        $dm->identity_type = $request->identity_type;
        $dm->branch_id = $request->branch_id;
        $dm->identity_image = $identity_image;
        $dm->image = $image_name;
        $dm->password = bcrypt($request->password);
        $dm->application_status = 'approved';
        $dm->approved_at = now();
        $dm->save();

        Toastr::success(translate('Delivery-man added successfully!'));
        return redirect('admin/delivery-man/list');
    }

    /**
     * @param $id
     * @return Application|Factory|View
     */
    public function edit($id): View|Factory|Application
    {
        $deliveryMan = $this->delivery_man->find($id);
        return view('admin-views.delivery-man.edit', compact('deliveryMan'));
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function status(Request $request): RedirectResponse
    {
        $deliveryMan = $this->delivery_man->find($request->id);
        $deliveryMan->is_active = $request->is_active;
        $deliveryMan->save();

        Toastr::success(translate('Delivery man status updated!'));
        return back();
    }

    /**
     * @param Request $request
     * @param $id
     * @return Application|RedirectResponse|Redirector
     */
    public function update(Request $request, $id): Redirector|RedirectResponse|Application
    {
        $check = $this->validateUploadedFile($request, ['image', 'identity_image']);
        if ($check !== true) {
            return $check;
        }

        $request->validate([
            'f_name' => 'required',
            'email' => 'required|regex:/(.+)@(.+)\.(.+)/i',
            'password_confirmation' => 'required_with:password|same:password',
            'image' => 'sometimes|image|max:' . $this->maxImageSizeKB . '|mimes:' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
            'identity_image' => 'sometimes|array',
            'identity_image.*' => 'image|max:' . $this->maxImageSizeKB . '|mimes:' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
        ], [
            'f_name.required' => 'First name is required!',
            'image.mimes' => 'Image must be a file of type: ' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
            'image.max' => translate('Image size must be below ' . $this->maxImageSizeReadable),
            'identity_image.*.mimes' => 'Identity image must be a file of type: ' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
            'identity_image.*.max' => translate('Identity image size must be below ' . $this->maxImageSizeReadable),
        ]);

        $deliveryMan = $this->delivery_man->find($id);

        if ($deliveryMan['email'] != $request['email']) {
            $request->validate([
                'email' => 'required|unique:delivery_men',
            ]);
        }

        if ($deliveryMan['phone'] != $request['phone']) {
            $request->validate([
                'phone' => ['required', 'unique:delivery_men', new Phone()],
            ]);
        }

        if (!empty($request->file('image'))) {
            $image_name = Helpers::update('delivery-man/', $deliveryMan->image, APPLICATION_IMAGE_FORMAT, $request->file('image'));
        } else {
            $image_name = $deliveryMan['image'];
        }

        if (!empty($request->file('identity_image'))) {
            foreach (json_decode($deliveryMan['identity_image'], true) as $img) {
                if (Storage::disk('public')->exists('delivery-man/' . $img)) {
                    Storage::disk('public')->delete('delivery-man/' . $img);
                }
            }
            $img_keeper = [];
            foreach ($request->identity_image as $img) {
                $identity_image = Helpers::upload('delivery-man/', APPLICATION_IMAGE_FORMAT, $img);
                $img_keeper[] = $identity_image;
            }
            $identity_image = json_encode($img_keeper);
        } else {
            $identity_image = $deliveryMan['identity_image'];
        }
        $deliveryMan->f_name = $request->f_name;
        $deliveryMan->l_name = $request->l_name;
        $deliveryMan->email = $request->email;
        $deliveryMan->phone = $request->phone;
        $deliveryMan->identity_number = $request->identity_number;
        $deliveryMan->identity_type = $request->identity_type;
        $deliveryMan->branch_id = $request->branch_id;
        $deliveryMan->identity_image = $identity_image;
        $deliveryMan->image = $image_name;
        $deliveryMan->password = strlen($request->password) > 1 ? bcrypt($request->password) : $deliveryMan['password'];
        $deliveryMan->save();
        Toastr::success(translate('Delivery-man updated successfully'));
        return redirect('admin/delivery-man/list');
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function delete(Request $request): RedirectResponse
    {
        $deliveryMan = $this->delivery_man->find($request->id);
        if (Storage::disk('public')->exists('delivery-man/' . $deliveryMan['image'])) {
            Storage::disk('public')->delete('delivery-man/' . $deliveryMan['image']);
        }

        foreach (json_decode($deliveryMan['identity_image'], true) as $img) {
            if (Storage::disk('public')->exists('delivery-man/' . $img)) {
                Storage::disk('public')->delete('delivery-man/' . $img);
            }
        }

        $deliveryMan->delete();
        Toastr::success(translate('Delivery-man removed!'));
        return back();
    }

    /**
     * @param Request $request
     * @return Application|Factory|View
     */
    public function pendingList(Request $request): Factory|View|Application
    {
        $perPage = (int)$request->query('per_page', Helpers::getPagination());
        $queryParams = ['per_page' => $perPage];

        $search = $request->query('search');

        // Start query
        $query = $this->delivery_man->with('branch')->where('application_status', 'pending');

        // Apply search filter
        if ($search) {
            $queryParams['search'] = $search;
            $query = $query->where(function ($q) use ($search) {
                $q->orWhere('f_name', 'like', "%{$search}%")
                    ->orWhere('l_name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $deliveryMan = $query->latest()->paginate($perPage)->appends($queryParams);

        return view('admin-views.delivery-man.pending-list', compact('deliveryMan', 'search', 'perPage'));
    }

    /**
     * @param Request $request
     * @return Application|Factory|View
     */
    public function deniedList(Request $request): Factory|View|Application
    {
        $perPage = (int)$request->query('per_page', Helpers::getPagination());
        $queryParams = ['per_page' => $perPage];

        $search = $request->query('search');

        // Start query
        $query = $this->delivery_man->with('branch')->where('application_status', 'denied');

        // Apply search filter
        if ($search) {
            $queryParams['search'] = $search;
            $query = $query->where(function ($q) use ($search) {
                $q->orWhere('f_name', 'like', "%{$search}%")
                    ->orWhere('l_name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $deliveryMan = $query->latest()->paginate($perPage)->appends($queryParams);

        return view('admin-views.delivery-man.denied-list', compact('deliveryMan', 'search', 'perPage'));
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function updateApplication(Request $request): RedirectResponse
    {
        $deliveryMan = $this->delivery_man->findOrFail($request->id);
        $deliveryMan->application_status = $request->status;
        $deliveryMan->save();

        try {
            $emailServices = Helpers::get_business_settings('mail_config');
            if (isset($emailServices['status']) && $emailServices['status'] == 1) {
                Mail::to($deliveryMan->email)->send(new \App\Mail\DMSelfRegistration($request->status, $deliveryMan->f_name . ' ' . $deliveryMan->l_name));
            }

        } catch (\Exception $ex) {
            info($ex);
        }

        Toastr::success(translate('application_status_updated_successfully'));
        return back();
    }

    public function details(Request $request, $id)
    {
        $branchIds = $this->order->where('delivery_man_id', $id)->pluck('branch_id')->toArray();
        $branches = $this->branch->whereIn('id', $branchIds)->get();
        $orders = $this->getDeliveryManOrderList($request, $id)
            ->paginate((int)($request->per_page ?? Helpers::pagination_limit()))
            ->appends($request->all());
        $deliveryMan = $this->delivery_man->with('orders')->where('id', $id)->first();

        return view('admin-views.delivery-man.details', compact('orders', 'branches', 'deliveryMan'));
    }

    public function export(Request $request)
    {
        $data = [];
        $deliveryMenList = $this->getDeliveryMenList($request)
            ->get()
            ->map(function($deliveryMan) use ($data) {
                $data['Name'] = $deliveryMan->f_name . ' ' . $deliveryMan->l_name;
                $data['Contact Info'] = $deliveryMan->phone . ' (' . $deliveryMan->email . ')';
                $data['Completed Orders'] = $deliveryMan->orders->where('order_status', 'delivered')->count();
                $data['Assigned Orders'] = $deliveryMan->orders->where('order_status', 'processing')->count();
                $data['Out For Delivery'] = $deliveryMan->orders->where('order_status', 'out_for_delivery')->count();
                $data['Status'] = $deliveryMan->is_active ? 'Active' : 'Inactive';

                return $data;
            });


        return (new FastExcel($deliveryMenList))->download('delivery-man-list.xlsx');
    }

    public function exportDetails(Request $request, $id)
    {
        $data = [];
        $deliveryMenOrdersList = $this->getDeliveryManOrderList($request, $id)
            ->get()
            ->map(function($order) use ($data) {
                $data['Order Id'] = $order->id;
                $data['Total items'] = $order->details->count();
                $data['Payment Method'] = $order->payment_method == 'multiple' ? implode(', ', $order->additional_payment_method ?? ["N/A"]) : translate($order->payment_method);
                $data['Branch'] = $order->branch->name;
                $data['Order Status'] = $order->order_status;

                return $data;
            });

        return (new FastExcel($deliveryMenOrdersList))->download('delivery-man-orders-list.xlsx');
    }

    private function getDeliveryMenList(Request $request)
    {
        $startDate = $endDate = '';
        if ($request->filled('joining_date')) {
            $joiningDate = explode(' - ', $request->joining_date);
            $startDate = Carbon::createFromFormat('d M Y', $joiningDate[0])->startOfDay()->format('Y-m-d H:i:s');
            $endDate = Carbon::createFromFormat('d M Y', $joiningDate[1])->endOfDay()->format('Y-m-d H:i:s');
        }

        return $this->delivery_man
            ->with(['branch', 'orders'])
            ->where('application_status', 'approved')
            ->when($request->filled('status') && in_array($request->status, ['0', '1'], true), fn ($q) => $q->where('is_active', (int) $request->status))
            ->when($request->filled('joining_date'), fn ($q) => $q->whereBetween('approved_at', [$startDate, $endDate]))
            ->when($request->filled('search'), fn ($q) => $q->where(function ($w) use ($request) {
                $w->whereRaw(
                    "CONCAT_WS(' ', f_name, l_name) LIKE ?",
                    ["%{$request->search}%"]
                )->orWhere('phone', 'like', "%{$request->search}%");
            }))
            ->when($request->filled('sort'), function ($q) use ($request) {
                match ($request->sort) {
                    'completed_order' => $q->withCount('getOrdersCompleted')
                        ->orderByDesc('get_orders_completed_count'),

                    'assigned_orders' => $q->withCount('getOrdersOnDelivery')
                        ->orderByDesc('get_orders_on_delivery_count'),

                    'newly_joined' => $q->orderByDesc('approved_at'),

                    default => null,
                };
            }
            )
            ->when(!$request->filled('sort'), fn ($q) => $q->withCount('getOrdersCompleted')
                ->orderByDesc('get_orders_completed_count'));
    }

    private function getDeliveryManOrderList(Request $request, $id)
    {
        $startDate = $endDate = '';
        if ($request->filled('order_date')) {
            $orderDate = explode(' - ', $request->order_date);
            $startDate = Carbon::createFromFormat('d M Y', $orderDate[0])->startOfDay()->format('Y-m-d H:i:s');
            $endDate = Carbon::createFromFormat('d M Y', $orderDate[1])->endOfDay()->format('Y-m-d H:i:s');
        }

        return $this->order
            ->with(['delivery_man', 'branch', 'details'])
            ->where('delivery_man_id', $id)
            ->when($request->filled('order_date'), fn ($q) => $q->whereBetween('created_at', [$startDate, $endDate]))
            ->when($request->filled('branch_id') && $request->branch_id != 'all', fn ($q) => $q->where('branch_id', $request->branch_id))
            ->when($request->filled('search'), fn($q) => $q->where(function ($w) use ($request) {
                $w->where('id', 'like' , "%{$request->search}%")
                     ->orwhereHas('branch', function ($y) use ($request) {
                         $y->where('name', 'like', "%{$request->search}%");
                     });
            }))
            ->latest();

    }
}
