<?php

namespace App\Http\Controllers\Admin;

use App\CentralLogics\Helpers;
use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\Conversation;
use App\Models\User;
use App\Traits\UploadSizeHelperTrait;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class ConversationController extends Controller
{
    use UploadSizeHelperTrait;

    public function __construct(
        private Admin        $admin,
        private Conversation $conversation,
        private User         $user
    )
    {
        $this->initUploadLimits();
    }

    /**
     * @return Application|Factory|View
     */
    public function list(): View|Factory|Application
    {
        $customerList = $this->conversation
            ->select('user_id', DB::raw('MAX(updated_at) as last_message_at'), DB::raw('SUM(CASE WHEN checked = 0 THEN 1 ELSE 0 END) as total_unread_message'))
            ->with('user')
            ->groupBy('user_id')
            ->orderBy('total_unread_message', 'desc')
            ->orderBy('last_message_at', 'desc')
            ->get();

        $conversations = DB::table('conversations')->latest()->get();
        return view('admin-views.messages.index', compact('conversations', 'customerList'));
    }

    /**
     * @param $user_id
     * @return JsonResponse
     */
    public function view($user_id): JsonResponse
    {
        $this->conversation
            ->where(['user_id' => $user_id, 'checked' => 0])
            ->update(['checked' => 1]);
        $user = $this->user
            ->where('id', $user_id)
            ->first();
        $userConversation = $this->conversation
            ->where(['user_id' => $user_id])
            ->orderBy('updated_at', 'asc')
            ->get();

        return response()->json([
            'view' => view('admin-views.messages.partials._conversation', compact('userConversation', 'user'))->render()
        ]);
    }

    /**
     * @param Request $request
     * @param $user_id
     * @return JsonResponse
     */
    public function store(Request $request, $user_id): JsonResponse
    {
        $check = $this->validateUploadedFile($request, ['images']);
        if ($check !== true) {
            return $check;
        }

        $validator = Validator::make($request->all(), [
            'reply' => 'required_without_all:images|nullable|string',
            'images' => 'sometimes|array',
            'images.*' => 'image|max:' . $this->maxImageSizeKB . '|mimes:' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
        ], [
            'images.*.mimes' => 'Image must be a file of type: ' . implode(',', array_column(IMAGE_EXTENSIONS, 'key')),
            'images.*.max' => translate('Image size must be below ' . $this->maxImageSizeReadable),
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => Helpers::error_processor($validator)]);
        }

        if ($request->images) {
            $id_img_names = [];
            foreach ($request->images as $img) {
                $image = Helpers::upload('conversation/', APPLICATION_IMAGE_FORMAT, $img);
                $image_url = $image;
                $id_img_names[] = $image_url;
            }
            $images = $id_img_names;
        } else {
            $images = null;
        }

        DB::table('conversations')->insert([
            'user_id' => $user_id,
            'reply' => $request->reply,
            'attachment' => isset($images) ? json_encode($images) : null,
            'checked' => 1,
            'is_reply' => true,
            'created_at' => now(),
            'updated_at' => now()
        ]);

        $userConversation = $this->conversation->where(['user_id' => $user_id])->get();
        $user = $this->user->find($user_id);

        $fcm_token = $user->cm_firebase_token;
        $data = [
            'title' => translate('New message arrived'),
            'description' => Str::limit($request->reply ?? '', 500),
            'order_id' => '',
            'image' => '',
            'type' => 'message',
        ];
        try {
            Helpers::send_push_notif_to_device($fcm_token, $data);
        } catch (\Exception $exception) {
            //
        }

        return response()->json([
            'success' => true,
            'message' => \App\CentralLogics\translate('Message sent'),
            'view' => view('admin-views.messages.partials._conversation', compact('userConversation', 'user'))->render()
        ]);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function updateFcmToken(Request $request): JsonResponse
    {
        try {
            $admin = $this->admin->find(auth('admin')->id());
            $admin->fcm_token = $request->fcm_token;
            $admin->save();

            return response()->json(['message' => 'FCM token updated successfully.'], 200);
        } catch (\Exception $exception) {
            return response()->json(['message' => 'FCM token updated failed.'], 200);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getConversations(Request $request): JsonResponse
    {
        $conversations = DB::table('conversations')->latest()->get();
        return response()->json([
            'conversation_sidebar' => view('admin-views.messages.partials._list', compact('conversations'))->render(),
        ]);
    }

    /**
     * @param Request $request
     * @return mixed|null
     */
    public function getFirebaseConfig(Request $request)
    {
        $config = Helpers::get_business_settings('firebase_message_config');
        return $config;
    }

    public function getCustomerList(Request $request)
    {
        $customerList = $this->conversation
            ->select('user_id', DB::raw('MAX(updated_at) as last_message_at'), DB::raw('SUM(CASE WHEN checked = 0 THEN 1 ELSE 0 END) as total_unread_message'))
            ->with('user')
            ->when($request->filled('search'), function ($query) use ($request) {
                $query->
                whereHas('user', function ($userQuery) use ($request) {
                    $userQuery->where(function ($subQuery) use ($request) {
                        $subQuery->where('f_name', 'like', "%{$request->search}%")
                            ->orWhere('l_name', 'like', "%{$request->search}%")
                            ->orWhere('phone', 'like', "%{$request->search}%");
                    });
                });
            })
            ->groupBy('user_id')
            ->orderBy('total_unread_message', 'desc')
            ->orderBy('last_message_at', 'desc')
            ->get();

        return response()->json([
            'view' => view('admin-views.messages.partials._customer-list', compact('customerList'))->render()
        ]);
    }

}
